
// const express = require('express');
// const fs = require('fs');
// const path = require('path');
// const bodyParser = require('body-parser');
// const cors = require('cors');
// const app = express();
// const PORT = process.env.PORT || 3001;
// const nodemailer = require('nodemailer')
// const multer = require('multer');
// const { error } = require('console');
// // Paths to data files
// const donationsFilePath = path.join(__dirname, 'data', 'donations.json');
// const settingsFilePath = path.join(__dirname, 'data', 'settings.json');

// app.use(bodyParser.json());
// app.use(cors());

// // Function to read donations
// const readDonations = () => {
//   try {
//     const data = fs.readFileSync(donationsFilePath, 'utf-8');
//     return JSON.parse(data);
//   } catch (error) {
//     console.error('Error reading donations file:', error);
//     return [];
//   }
// };


// ////////////////////File upload portion//////////////////////////////////////////////////////////
// const uploadsDir = path.join(__dirname, 'uploads');

// if (!fs.existsSync(uploadsDir)) {
//   fs.mkdirSync(uploadsDir, { recursive: true });
// }

// const storage = multer.diskStorage({
//   destination: (req, file, cb) => cb(null, uploadsDir),
//   filename: (req, file, cb) => {
//     const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
//     const ext = path.extname(file.originalname);
//     cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
//   },
// });
// const upload = multer({ storage });

// // serve uploads folder statically
// app.use('/uploads', express.static(uploadsDir));


// app.post('/api/upload/image', upload.single('image'), (req, res) => {
//   if (!req.file) {
//     return res.status(400).json({ message: 'No image uploaded' });
//   }

//   // return relative path (frontend can use this)
//   const imagePath = `/uploads/${req.file.filename}`;
//   res.status(200).json({ imagePath });
// });

// // POST route for banner video upload
// app.post('/api/upload/video', upload.single('video'), (req, res) => {
//   if (!req.file) {
//     return res.status(400).json({ message: 'No video uploaded' });
//   }

//   const videoPath = `/uploads/${req.file.filename}`;
//   res.status(200).json({ videoPath });
// });

// /////////////////////////////////////////////////////////////////////////////////////////////////

// // Function to write donations
// const writeDonations = (donations) => {
//   try {
//     fs.writeFileSync(donationsFilePath, JSON.stringify(donations, null, 2));
//   } catch (error) {
//     console.error('Error writing donations file:', error);
//   }
// };

// // GET all donations
// app.get('/api/donations', (req, res) => {
//   const donations = readDonations();
//   res.status(200).json(donations);

// });

// app.post("/api/donations", async (req, res) => {
//   const { name, amount, notes, email } = req.body;

//   if (!name || !amount || !email) {
//     return res.status(400).json({ message: "Missing required fields" });
//   }

//   const newDonation = {
//     id: Date.now().toString(),
//     name,
//     amount,
//     notes: notes || "",
//     email,
//     timestamp: new Date().toISOString(),
//   };

//   const donations = readDonations();
//   donations.push(newDonation);
//   writeDonations(donations);

//   const emailConfig = readSettings();
//   if (!emailConfig) {
//     return res.status(500).json({ message: "Error loading email settings" });
//   }

//   const transporter = nodemailer.createTransport({
//     host: "smtp.gmail.com",
//     port: 587,
//     secure: false,
//     auth: {
//       user: emailConfig.NotificationfromEmail,
//       pass: emailConfig.NotificationfromEmailPassword,  // Use Gmail App Password
//     },
//   });
//   // Send mail
//   try {
//     await transporter.sendMail({
//       from: `"Donation Portal" <${emailConfig.senderEmail}>`,
//       to: emailConfig.NotificationToEmail, // recipient
//       subject: "🎉 New Donation Received!",
//       html: `
//         <h2>New Donation Received 🎉</h2>
//         <p><b>Name:</b> ${name}</p>
//         <p><b>Email:</b> ${email}</p>
//         <p><b>Amount:</b> $${amount}</p>
//         <p><b>Notes:</b> ${notes || "None"}</p>
//       `,
//     });

//     res.status(201).json({ message: "Donation added and email sent successfully" });
//   } catch (err) {
//     console.error("Error sending email:", err);
//     res.status(500).json({ message: "Donation saved but failed to send email" });
//   }
// });

// // Function to read settings
// const readSettings = () => {
//   try {
//     const data = fs.readFileSync(settingsFilePath, 'utf-8');
//     return JSON.parse(data);
//   } catch (error) {
//     console.error('Error reading settings file:', error);
//     return {};
//   }
// };

// // Function to write settings
// const writeSettings = (settings) => {
//   try {
//     fs.writeFileSync(settingsFilePath, JSON.stringify(settings, null, 2));
//   } catch (error) {
//     console.error('Error writing settings file:', error);
//   }
// };

// // GET settings
// app.get('/api/settings', (req, res) => {
//   const settings = readSettings();
//   res.status(200).json(settings);
// });

// // POST update settings (no strict validation to allow all fields)
// app.post('/api/settings', (req, res) => {
//   const newSettings = req.body;
//   // Merge with existing to avoid overwriting unrelated fields
//   try {
//     const currentSettings = readSettings();
//     const updatedSettings = { ...currentSettings, ...newSettings };
//     writeSettings(updatedSettings);

//     return res.status(200).json({
//       message: 'Settings updated successfully',
//       updatedSettings
//     });
//   } catch (err) {
//     console.error('Error updating settings:', err);
//     return res.status(500).json({ message: 'Internal server error', error: err.message });
//   }
// });

// app.post('/api/clear',(req,res) =>{
//   try {
//     writeDonations([])

//     return res.status(200).json({
//       message:"successfully cleared the supporters data",
//     });
//   }catch(err){
//     console.error('Error clearing the supporters data',err);
//     return res.status(500).json({
//       message:'Internal server error',error:err.message
//     });
//   }
// })

// app.listen(PORT, () => {
//   console.log(`Server running at http://localhost:${PORT}`);
// });



// server.js
const express = require("express");
const fs = require("fs");
const path = require("path");
const bodyParser = require("body-parser");
const cors = require("cors");
const multer = require("multer");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const nodemailer = require("nodemailer");

const app = express();
const PORT = process.env.PORT || 3001;
const JWT_SECRET = process.env.ADMIN_JWT_SECRET || "dev-secret-change-me"; // change this in production

app.use(bodyParser.json());
app.use(cors());

// -------------------- Data paths --------------------
const dataDir = path.join(__dirname, "data");
if (!fs.existsSync(dataDir)) fs.mkdirSync(dataDir, { recursive: true });

const donationsFilePath = path.join(dataDir, "donations.json");
const settingsFilePath = path.join(dataDir, "settings.json");
const adminConfigPath = path.join(dataDir, "adminConfig.json");

// -------------------- Upload config (multer) --------------------
const uploadsDir = path.join(__dirname, "uploads");
if (!fs.existsSync(uploadsDir)) fs.mkdirSync(uploadsDir, { recursive: true });

const storage = multer.diskStorage({
  destination: (req, file, cb) => cb(null, uploadsDir),
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    const ext = path.extname(file.originalname);
    cb(null, `${file.fieldname}-${uniqueSuffix}${ext}`);
  },
});
const upload = multer({ storage });
app.use("/uploads", express.static(uploadsDir));

// -------------------- Helpers --------------------
const readJSON = (filePath, defaultValue) => {
  try {
    if (!fs.existsSync(filePath)) return defaultValue;
    const raw = fs.readFileSync(filePath, "utf8");
    return JSON.parse(raw);
  } catch (err) {
    console.error("readJSON error:", err);
    return defaultValue;
  }
};
const writeJSON = (filePath, obj) => {
  try {
    fs.writeFileSync(filePath, JSON.stringify(obj, null, 2), "utf8");
  } catch (err) {
    console.error("writeJSON error:", err);
  }
};

// -------------------- Admin config helpers --------------------
const readAdminConfig = () => readJSON(adminConfigPath, null);
const writeAdminConfig = (obj) => writeJSON(adminConfigPath, obj);

// -------------------- Admin endpoints --------------------

// Setup initial admin - Only allowed if adminConfig.json does NOT exist
app.post("/api/admin/setup", (req, res) => {
  try {
    if (fs.existsSync(adminConfigPath)) {
      return res.status(400).json({ message: "Admin already set up" });
    }
    const { username, password } = req.body;
    if (!username || !password) return res.status(400).json({ message: "username & password required" });
    const hash = bcrypt.hashSync(password, 10);
    writeAdminConfig({ adminUsername: username, adminPasswordHash: hash });
    return res.json({ message: "Admin created" });
  } catch (err) {
    console.error("admin setup error:", err);
    return res.status(500).json({ message: "Internal server error" });
  }
});

// Login -> returns JWT token
app.post("/api/admin/login", (req, res) => {
  try {
    const { username, password } = req.body;
    const cfg = readAdminConfig();
    if (!cfg) {
      return res.status(400).json({ message: "Admin not configured. Run /api/admin/setup first." });
    }
    if (username !== cfg.adminUsername) return res.status(401).json({ message: "Invalid credentials" });

    const match = bcrypt.compareSync(password, cfg.adminPasswordHash);
    if (!match) return res.status(401).json({ message: "Invalid credentials" });

    const token = jwt.sign({ username }, JWT_SECRET, { expiresIn: "8h" });
    return res.json({ token });
  } catch (err) {
    console.error("admin login error:", err);
    return res.status(500).json({ message: "Internal server error" });
  }
});

// Middleware to protect endpoints
function authenticateToken(req, res, next) {
  const auth = req.headers["authorization"];
  if (!auth) return res.status(401).json({ message: "Missing authorization header" });
  const parts = auth.split(" ");
  if (parts.length !== 2 || parts[0] !== "Bearer") return res.status(401).json({ message: "Invalid auth header" });

  const token = parts[1];
  try {
    const payload = jwt.verify(token, JWT_SECRET);
    req.admin = payload;
    next();
  } catch (err) {
    return res.status(401).json({ message: "Invalid or expired token" });
  }
}

// Protected endpoint to change admin password
app.post("/api/admin/change-password", authenticateToken, (req, res) => {
  try {
    const { oldPassword, newPassword } = req.body;
    if (!oldPassword || !newPassword) return res.status(400).json({ message: "oldPassword and newPassword required" });

    const cfg = readAdminConfig();
    if (!cfg) return res.status(400).json({ message: "Admin not configured" });

    const match = bcrypt.compareSync(oldPassword, cfg.adminPasswordHash);
    if (!match) return res.status(401).json({ message: "Old password is incorrect" });

    const newHash = bcrypt.hashSync(newPassword, 10);
    cfg.adminPasswordHash = newHash;
    writeAdminConfig(cfg);

    return res.json({ message: "Password updated successfully" });
  } catch (err) {
    console.error("change-password error:", err);
    return res.status(500).json({ message: "Internal server error" });
  }
});

app.get("/",(req,res)=>{
  return res.send("server Started")
})


// Return current admin (from token)
app.get("/api/admin/me", authenticateToken, (req, res) => {
  return res.json({ username: req.admin.username });
});

// -------------------- Upload endpoints (image/video) --------------------
app.post("/api/upload/image", upload.single("image"), (req, res) => {
  if (!req.file) return res.status(400).json({ message: "No image uploaded" });
  const imagePath = `/uploads/${req.file.filename}`;
  res.json({ imagePath });
});
app.post("/api/upload/video", upload.single("video"), (req, res) => {
  if (!req.file) return res.status(400).json({ message: "No video uploaded" });
  const videoPath = `/uploads/${req.file.filename}`;
  res.json({ videoPath });
});

// -------------------- Donations read/write --------------------
const readDonations = () => readJSON(donationsFilePath, []);
const writeDonations = (donations) => writeJSON(donationsFilePath, donations);

app.get("/api/donations", (req, res) => {
  const donations = readDonations();
  res.json(donations);
});

app.post("/api/donations", async (req, res) => {
  try {
    const { name, amount, notes, email } = req.body;
    if (!name || !amount || !email) return res.status(400).json({ message: "Missing required fields" });

    const newDonation = {
      id: Date.now().toString(),
      name,
      amount,
      notes: notes || "",
      email,
      timestamp: new Date().toISOString(),
    };

    const donations = readDonations();
    donations.push(newDonation);
    writeDonations(donations);

    // Email sending (best-effort)
    const emailConfig = readJSON(settingsFilePath, {});
    if (!emailConfig || !emailConfig.NotificationfromEmail || !emailConfig.NotificationfromEmailPassword || !emailConfig.NotificationToEmail) {
      // Saved donation but skip email if not configured
      return res.status(201).json({ message: "Donation saved (email not configured)" });
    }

    const transporter = nodemailer.createTransport({
      host: "smtp.gmail.com",
      port: 587,
      secure: false,
      auth: {
        user: emailConfig.NotificationfromEmail,
        pass: emailConfig.NotificationfromEmailPassword,
      },
    });

    try {
      await transporter.sendMail({
        from: `"Donation Portal" <${emailConfig.NotificationfromEmail}>`,
        to: emailConfig.NotificationToEmail,
        subject: "🎉 New Donation Received!",
        html: `
          <h2>New Donation Received 🎉</h2>
          <p><b>Name:</b> ${name}</p>
          <p><b>Email:</b> ${email}</p>
          <p><b>Amount:</b> $${amount}</p>
          <p><b>Notes:</b> ${notes || "None"}</p>
        `,
      });
      return res.status(201).json({ message: "Donation added and email sent successfully" });
    } catch (mailErr) {
      console.error("Error sending email:", mailErr);
      return res.status(201).json({ message: "Donation saved but failed to send email" });
    }
  } catch (err) {
    console.error("donation POST error:", err);
    return res.status(500).json({ message: "Internal server error" });
  }
});

// -------------------- Settings read/write --------------------
app.get("/api/settings", (req, res) => {
  const settings = readJSON(settingsFilePath, {});
  res.json(settings);
});

app.post("/api/settings", (req, res) => {
  try {
    const newSettings = req.body || {};
    const current = readJSON(settingsFilePath, {});
    const updated = { ...current, ...newSettings };
    writeJSON(settingsFilePath, updated);
    return res.json({ message: "Settings updated", updatedSettings: updated });
  } catch (err) {
    console.error("settings POST error:", err);
    return res.status(500).json({ message: "Internal server error" });
  }
});

// -------------------- Clear donations --------------------
app.post("/api/clear", (req, res) => {
  try {
    writeDonations([]);
    return res.json({ message: "Supporters data cleared" });
  } catch (err) {
    console.error("clear error:", err);
    return res.status(500).json({ message: "Internal server error" });
  }
});

// -------------------- Start server --------------------
app.listen(PORT, () => {
  console.log(`Server running at http://localhost:${PORT}`);
  // If adminConfig missing, print instruction (we DON'T auto-create here)
  if (!fs.existsSync(adminConfigPath)) {
    console.warn(
      `⚠️  Admin config not found at ${adminConfigPath}. Create it with /api/admin/setup or place adminConfig.json in data/.`
    );
  }
});
